/*
 * node_test.c
 * Test node handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <string.h>

#include "main.h"

#include "bt_appl/common_def.h"
#include "bt_appl_dummy.h"
#include "callbacks.h"
#include "utils_async_queue.h"
#include "utils_test_queue.h"
#include "node_test.h"

#include "dbus_error.h"
#include "debug.h"

#ifdef USE_TEST_NODE

/************/
/* #defines */
/************/

/********************/
/* TEST DEFINITIONS */
/********************/

/* OUR TEST OBJECT DEFINITION */
typedef struct {
	/* The parent class object state. */
	GObject parent;

	struct test_queue_list tql_cfm;

} TestObject;

/* OUR CLASS DEFINITION */
typedef struct {
	/* The parent class state. */
	GObjectClass parent;

	/* put test signal stuff here, (if any) */

} TestObjectClass;

/*
 * MAGIC here to define our test object functions etc. in GLib
 * declares test_object_init(), test_object_class_init(),
 * test_object_get_type()
 */
G_DEFINE_TYPE(TestObject, test_object, G_TYPE_OBJECT)

/*
 * Forward declaration of the function that will return the GType of
 * the Value implementation. Not used
 */
GType test_object_get_type(void);

/*
 * Macro for the above. It is common to define macros using the
 * naming convention (seen below) for all GType implementations.
 */
#define TEST_TYPE_OBJECT	(test_object_get_type())

/* macros to get to and from the class to object */
#define TEST_OBJECT(object) \
	(G_TYPE_CHECK_INSTANCE_CAST ((object), \
	TEST_TYPE_OBJECT, TestObject))
#define TEST_OBJECT_CLASS(klass) \
	(G_TYPE_CHECK_CLASS_CAST ((klass), \
	TEST_TYPE_OBJECT, TestObjectClass))
#define TEST_IS_OBJECT(object) \
	(G_TYPE_CHECK_INSTANCE_TYPE ((object), \
	TEST_TYPE_OBJECT))
#define TEST_IS_OBJECT_CLASS(klass) \
	(G_TYPE_CHECK_CLASS_TYPE ((klass), \
	TEST_TYPE_OBJECT))
#define TEST_OBJECT_GET_CLASS(obj) \
	(G_TYPE_INSTANCE_GET_CLASS ((obj), \
	TEST_TYPE_OBJECT, TestObjectClass))


/******************/
/* TEST CALLBACKS */
/******************/

/*
 * callback handling, parameters needs to match marshalling signatures
 *
 * callbacks for the auto generated method protoypes
 * Forward declare the prototypes of the callback method handlers for the
 * dbus_glib_adapter_object_methods[]
 */

gboolean test_object_test_use_dummy_api(TestObject *object,
						gboolean use_dummy_api_in,
						GError **error);

gboolean test_object_test_show_dummy_api(TestObject *object,
						GError **error);

gboolean test_object_test_get_dummy_api(TestObject *object,
						GPtrArray **settings_out,
						GError **error);

gboolean test_object_test_set_dummy_api(TestObject *object,
						gboolean use_func_name_in,
						guint32 func_id_in,
						gchar *func_name_in,
						gboolean use_dummy_api_in,
						gboolean use_alps_error_name_in,
						guint32 alps_error_id_in,
						gchar *alps_error_name_in,
						GError **error);

gboolean test_object_test_free_cfm_data(TestObject *object,
						GError **error);

gboolean test_object_test_seg_fault_report(TestObject *object,
						GError **error);

gboolean test_object_test_assert_fault_report(TestObject *object,
						GError **error);

gboolean test_object_test_bt_appl_phone_status_ind(TestObject *object,
					GByteArray *addr_in,
					guint8 phone_service_in,
					guint8 phone_signal_in,
					guint8 phone_roaming_in,
					guint8 phone_battery_in,
					GByteArray *phone_operator_name_in,
					guint32 phone_supp_feature_in,
					guint8 phone_AgThreeWayCallSuppFeatures,
					GError **error);

gboolean test_object_test_bt_appl_pb_dl_start_cfm(TestObject *object,
						GByteArray *addr_in,
						gboolean use_dummy_api_in,
						guint32 dbus_error_in,
						guint8 status_in,
						guint32 pb_number_in,
						guint8 new_missed_call_in,
						GError **error);

gboolean test_object_test_bt_appl_pb_dl_data_ind(TestObject *object,
							GByteArray *addr_in,
							guint8 data_type_in,
							GByteArray *data_in,
							GError **error);

gboolean test_object_test_bt_appl_pb_dl_data_complete_ind(TestObject *object,
							GByteArray *addr_in,
							guint8 status_in,
							guint32 pb_number_in,
							GError **error);

gboolean test_object_test_bt_appl_pb_dl_calendar_complete_ind(
							TestObject *object,
							GByteArray *addr_in,
							guint8 status_in,
							GError **error);

gboolean test_object_test_bt_appl_pb_dl_status_chg_ind(TestObject *object,
							GByteArray *addr_in,
							guint8 status_in,
							GError **error);

gboolean test_object_test_bt_appl_pb_dl_con_lost_ind(TestObject *object,
							GByteArray *addr_in,
							GError **error);

gboolean test_object_test_map_list_msg_data(TestObject *object,
						GByteArray *addr_in,
						gchar *service_name_in,
						GByteArray *handle_in,
						gchar *subject_in,
						gchar *data_time_in,
						gchar *sender_name_in,
						gchar *sender_address_in,
						gchar *reply_to_address_in,
						gchar *recipient_name_in,
						gchar *recipient_address_in,
						guint8 type_in,
						guint32 size_in,
						guint8 text_in,
						guint8 reception_status_in,
						guint32 attach_size_in,
						guint8 priority_in,
						guint8 read_in,
						guint8 sent_in,
						guint8 protect_in,
						GError **error);

gboolean test_object_test_map_list_msg_data_complete(TestObject *object,
							GByteArray *addr_in,
							gchar *service_name_in,
							guint32 size_in,
							guint8 status_in,
							GError **error);

gboolean test_object_test_bt_appl_avp_get_metadata_attr_cfm(
						TestObject *object,
						GByteArray *addr_in,
						gboolean use_dummy_api_in,
						guint8 dbus_error_in,
						guint8 status_in,
						guint8 num_attributes_in,
						GByteArray *data_in,
						GError **error);

gboolean test_object_test_avp_get_media_player_item_cfm(TestObject *object,
						GByteArray *addr_in,
						gboolean use_dummy_api_in,
						guint8 dbus_error_in,
						guint8 status_in,
						guint32 num_items,
						GError **error);

gboolean test_object_test_avp_get_media_player_item_ind(TestObject *object,
						GByteArray *addr_in,
						guint8 status_in,
						GByteArray *data_in,
						GError **error);

gboolean test_object_test_bt_appl_authentication_ind(TestObject *object,
						GByteArray *addr_in,
						GError **error);

/* include auto-generated binding information */
#include "alps-bt-test-introspection.h"


/***********************************/
/* OBJECT AND CLASS INITIALIZATION */
/***********************************/


/*
 * TestObject object initialiser function
 *
 * Used to initialise each object instance via g_object_new()
 */
static void test_object_init(TestObject *object)
{
	DEBUG_FUNC("Called");
	g_assert(object != NULL);

	/* add code to initialise created object contents */
}

/*
 * TestObject class initialiser function
 *
 * Used to initialise each class via the first call of g_object_new()
 */
static void test_object_class_init(TestObjectClass *klass)
{
#ifdef NOT_USED
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
#endif

	DEBUG_FUNC("Called");
	g_assert(klass != NULL);

	/* create new signals here, if needed */

	DEBUG("Binding our GObject with D-bus...");
	/* Uses auto-generated methods and signal structure */
	dbus_g_object_type_install_info(TEST_TYPE_OBJECT,
		&dbus_glib_test_object_object_info);

}


/* LOCAL GLOBAL */
/* test object pointer */
static TestObject *test_object = NULL;


/*
 * create a new object
 *
 * Inputs:
 * bus = point to the bus proxy
 * Exported function
 */
gboolean test_object_creation(DBusGConnection *bus, gchar *node_name)
{
	DEBUG("Creating an instance of the TestObject"
 				" and calling initialise functions...");

	test_object = g_object_new(TEST_TYPE_OBJECT, NULL);
	if(test_object == NULL) {
		DEBUG_ERROR("Failed to create an instance of a TestObject");
		return FALSE;
	}

	utils_test_queue_initialise(&test_object->tql_cfm);

#ifdef USE_DUMMY_API
	/* export the test queue list to the DUMMY_API */
	dummy_set_test_node_queue(&test_object->tql_cfm);
#endif

	DEBUG("Registering the test GObject onto the D-Bus");

	/* the object's path does not come from the XML file */
	dbus_g_connection_register_g_object(bus,
		node_name,	/* Object's path for the test node */
		G_OBJECT(test_object));

	return TRUE;
}


/*****************************/
/* METHOD CALLBACK FUNCTIONS */
/*****************************/

/*
 * Method call for "TestUseDummyApi"
 */
gboolean test_object_test_use_dummy_api(TestObject *object,
						gboolean use_dummy_api_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG("use all DUMMY_API functions: %s",
		use_dummy_api_in == TRUE ? "yes" : "no");

#ifdef USE_DUMMY_API
	/* change the DUMMY_API setting */
	dummy_use_all_dummy_api_functions(use_dummy_api_in);
#else
	/* spoof an ALPS error code as the DUMMY_API is not compiled in */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
						"Test_Use_Dummy_API",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestShowDummyApi"
 */
gboolean test_object_test_show_dummy_api(TestObject *object,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

#ifdef USE_DUMMY_API
	/* show the DUMMY_API setting */
	dummy_init_scenarios();
#else
	/* spoof an ALPS error code as the DUMMY_API is not compiled in */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
						"Test_Show_Dummy_API",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

#define DBUS_STRUCT_UINT_STRING_UCHAR_UINT_STRING (dbus_g_type_get_struct ("GValueArray", \
	G_TYPE_UINT, G_TYPE_STRING, G_TYPE_UCHAR, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID))

/*
 * Method call for "TestGetDummyApi"
 */
gboolean test_object_test_get_dummy_api(TestObject *object,
						GPtrArray **settings_out,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
#ifdef USE_DUMMY_API
	enum function_number loop;
#endif

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

#ifdef USE_DUMMY_API
	g_assert(settings_out != NULL);
	*settings_out = g_ptr_array_sized_new(FUNC_total);
	g_assert(*settings_out != NULL);

	for (loop = FUNC_Initialization_REQ; loop < FUNC_total; loop++) {
		GValue *value;
		gchar *func_name;
		gchar *dup_func_name = NULL;
		gchar *error_code_str;
		gchar *dup_error_code_str = NULL;

		value = g_new0(GValue, 1);
		g_value_init(value, DBUS_STRUCT_UINT_STRING_UCHAR_UINT_STRING);
		g_value_set_boxed_take_ownership(value,
			dbus_g_type_specialized_construct(DBUS_STRUCT_UINT_STRING_UCHAR_UINT_STRING));

		func_name = dummy_get_func_name(loop);
		if (func_name != NULL) {
			dup_func_name = g_strndup((gchar *) func_name,
							strlen(func_name));
		}

		error_code_str = lookup_bthresult_str(dummy_get_return_code(loop));
		if (error_code_str != NULL) {
			dup_error_code_str = g_strndup((gchar *) error_code_str,
							strlen(error_code_str));
		}

		dbus_g_type_struct_set(value,
			0, loop,
			1, dup_func_name,
			2, dummy_get_dummy_state(loop),
			3, dummy_get_return_code(loop),
			4, dup_error_code_str,
			G_MAXUINT);

		g_ptr_array_add(*settings_out, g_value_get_boxed(value));
		g_free(value);
		g_free(dup_func_name);
		g_free(dup_error_code_str);
	}
#else
	/* spoof an ALPS error code as the DUMMY_API is not compiled in */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
						"Test_Get_Dummy_API",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestSetDummyApi"
 */
gboolean test_object_test_set_dummy_api(TestObject *object,
						gboolean use_func_name_in,
						guint32 func_id_in,
						gchar *func_name_in,
						gboolean use_dummy_api_in,
						gboolean use_alps_error_name_in,
						guint32 alps_error_id_in,
						gchar *alps_error_name_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_ERROR_PARAMETERS;
	gboolean return_code = FALSE;
#ifdef USE_DUMMY_API
	enum function_number loop;
	gboolean good_parms = TRUE;
#endif

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

#ifdef USE_DUMMY_API
	if (use_func_name_in == TRUE) {
		/* find the DUMMY API function number */
		good_parms = FALSE;
		for (loop = FUNC_Initialization_REQ; loop < FUNC_total; loop++) {
			if (g_strcmp0(func_name_in, dummy_get_func_name(loop)) == 0) {
				func_id_in = loop;
				good_parms = TRUE;
				break;
			}
		}
	}

	if (use_alps_error_name_in == TRUE) {
		/* find the Alps error code */
		good_parms = FALSE;
		for (loop = BT_APPL_SUCCESS; loop <= BT_APPL_ERROR_SHUTDOWN_IS_ONGOING; loop++) {
			if (g_strcmp0(alps_error_name_in, lookup_bthresult_str(loop)) == 0) {
				alps_error_id_in = loop;
				good_parms = TRUE;
				break;
			}
		}
	}

	if (func_id_in >= FUNC_total) {
		good_parms = FALSE;
	}

	if (good_parms == TRUE) {
		/* set the DUMMY_API setting */
		dummy_change_dummy_api_settings(func_id_in,
						use_dummy_api_in,
						alps_error_id_in);
		result = BT_APPL_SUCCESS;
	}			
#else
	/* spoof an ALPS error code as the DUMMY_API is not compiled in */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
						"Test_Set_Dummy_API",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestFreeCfmData"
 */
gboolean test_object_test_free_cfm_data(TestObject *object,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/* free the CFM test data queue */
	utils_test_queue_destroy(&object->tql_cfm);

	return_code = report_any_error_to_dbus(result,
						"Test_Show_Dummy_API",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestSegFaultReport"
 */
gboolean test_object_test_seg_fault_report(TestObject *object,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
	char **crash_ptr = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG_ERROR("Deliberate test of segmentation fault handling...");
	DEBUG_ERROR("%s", *crash_ptr);
	/* Crash here */

	return_code = report_any_error_to_dbus(result,
						"Test_Seg_Fault_Report",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestAssertFaultReport"
 */
gboolean test_object_test_assert_fault_report(TestObject *object,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
	char **crash_ptr = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG_ERROR("Deliberate test of assert fault handling...");

	g_assert(crash_ptr != NULL);
	/* Crash here */

	return_code = report_any_error_to_dbus(result,
						"Test_Assert_Fault_Report",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestBtApplPhoneStatusInd"
 */
gboolean test_object_test_bt_appl_phone_status_ind(TestObject *object,
					GByteArray *addr_in,
					guint8 phone_service_in,
					guint8 phone_signal_in,
					guint8 phone_roaming_in,
					guint8 phone_battery_in,
					GByteArray *phone_operator_name_in,
					guint32 phone_supp_feature_in,
					guint8 phone_AgThreeWayCallSuppFeatures,
					GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);
	g_assert(phone_operator_name_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {
		PhoneStatus_t phone_status;

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		/* put the variables in the phone status structure */
		phone_status.ucService = phone_service_in;
		phone_status.ucSignal = phone_signal_in;
		phone_status.ucRoaming = phone_roaming_in;
		phone_status.ucBattery = phone_battery_in;
		phone_status.pucOperatorName = (u8 *)
			g_strndup((gchar *) &phone_operator_name_in->data[0],
					phone_operator_name_in->len);
		phone_status.ulAGsuppfeature = phone_supp_feature_in;
		phone_status.ucAgThreeWayCallSuppFeatures = phone_AgThreeWayCallSuppFeatures;

		/* call the callback handler function */
		result = BT_APPL_Phone_Status_IND_Callback_DUMMY(&addr_in->data[0],
								&phone_status);
		/* free the duplicated string */
		g_free(phone_status.pucOperatorName);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_PBDL_Calendar_Complete_IND",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestBtApplPbDlStartCfm"
 */
gboolean test_object_test_bt_appl_pb_dl_start_cfm(TestObject *object,
						GByteArray *addr_in,
						gboolean use_dummy_api_in,
						guint32 dbus_error_in,
						guint8 status_in,
						guint32 pb_number_in,
						guint8 new_missed_call_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

#ifdef USE_DUMMY_API

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	if (result == BT_APPL_SUCCESS) {

		/* set individual DUMMY_API settings */

		DEBUG("use DUMMY_API function: %s",
			use_dummy_api_in == TRUE ? "yes" : "no");

		DEBUG("set D-bus error to: %s",
			lookup_bthresult_str((BTHRESULT) dbus_error_in));

		/* change the DUMMY_API setting */
		dummy_change_dummy_api_settings(FUNC_PBDL_Start_REQ,
							use_dummy_api_in,
							dbus_error_in);
	}

	if ((result == BT_APPL_SUCCESS) && (use_dummy_api_in == TRUE)) {
		/* only save the test data when the DUMMY API is being used */

		struct pbdl_start_cfm_parms_s *cfm_test_data;

		/* create a record */
		cfm_test_data = (struct pbdl_start_cfm_parms_s *)
				g_malloc0(sizeof(struct pbdl_start_cfm_parms_s *));

		g_assert(cfm_test_data != NULL);

		/* copy the test parameter values into the record */
		cfm_test_data->status = status_in;
		cfm_test_data->pb_number = pb_number_in;
		cfm_test_data->new_missed_call = new_missed_call_in;

		utils_test_queue_add_list_entry(&object->tql_cfm,
						ID_PBDL_Start_CFM,
						&addr_in->data[0],
						cfm_test_data);
	}

#else
	/*
         * spoof an ALPS error code as the DUMMY_API is not compiled in
	 */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_PBDL_Start",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestBtApplPbDlDataInd"
 */
gboolean test_object_test_bt_appl_pb_dl_data_ind(TestObject *object,
							GByteArray *addr_in,
							guint8 data_type_in,
							GByteArray *data_in,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);
	g_assert(data_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		/* call the callback handler function */
		result = BT_APPL_PBDL_Data_IND_Callback(*device_address,
					 		data_type_in,
					 		data_in->len,
				 			&data_in->data[0]);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
						"Test_BT_APPL_PBDL_Data_IND",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call for "TestBtApplPbDlDataCompleteInd"
 */
gboolean test_object_test_bt_appl_pb_dl_data_complete_ind(TestObject *object,
							GByteArray *addr_in,
							guint8 status_in,
							guint32 pb_number_in,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		/* call the callback handler function */
		result = BT_APPL_PBDL_Data_Complete_IND_Callback(
								*device_address,
					 			status_in,
					 			pb_number_in);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_PBDL_Data_Complete_IND",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call for "TestBtApplPbDlCalendarCompleteInd"
 */
gboolean test_object_test_bt_appl_pb_dl_calendar_complete_ind(
							TestObject *object,
							GByteArray *addr_in,
							guint8 status_in,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		/* call the callback handler function */
		result = BT_APPL_PBDL_Calendar_Complete_IND_Callback(
								*device_address,
					 			status_in);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_PBDL_Calendar_Complete_IND",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call for "TestBtApplPbDlConLostInd"
 */
gboolean test_object_test_bt_appl_pb_dl_con_lost_ind(TestObject *object,
							GByteArray *addr_in,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		/* call the callback handler function */
		result = BT_APPL_PBDL_Con_Lost_IND_Callback(*device_address);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_PBDL_Con_Lost_IND",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call for "TestBtApplPbDlStatusChgInd"
 */
gboolean test_object_test_bt_appl_pb_dl_status_chg_ind(TestObject *object,
							GByteArray *addr_in,
							guint8 status_in,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		/* call the callback handler function */
		result = BT_APPL_PBDL_Status_Chg_IND_Callback(*device_address,
					 			status_in);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_PBDL_Status_Chg_IND",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call for "TestMapListMsgData"
 */
gboolean test_object_test_map_list_msg_data(TestObject *object,
						GByteArray *addr_in,
						gchar *service_name_in,
						GByteArray *handle_in,
						gchar *subject_in,
						gchar *data_time_in,
						gchar *sender_name_in,
						gchar *sender_address_in,
						gchar *reply_to_address_in,
						gchar *recipient_name_in,
						gchar *recipient_address_in,
						guint8 type_in,
						guint32 size_in,
						guint8 text_in,
						guint8 reception_status_in,
						guint32 attach_size_in,
						guint8 priority_in,
						guint8 read_in,
						guint8 sent_in,
						guint8 protect_in,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;
	MsgListingPtr_t msg_listing = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		/* create a message listing record */
		msg_listing = (MsgListingPtr_t)
				g_malloc0(sizeof(MsgListing_t));

		g_assert(msg_listing != NULL);

		DEBUG("service_name %s", service_name_in);

		/* fill the message listing record */
		msg_listing->pucHandle = &handle_in->data[0];
		msg_listing->ucHandleLen = handle_in->len;

		msg_listing->pucSubject = (guchar *) subject_in,
		msg_listing->ucSubjectLen = strlen(subject_in);

		msg_listing->pucDatetime = (guchar *) data_time_in,
		msg_listing->ucDatetimeLen = strlen(data_time_in);

		msg_listing->pucSenderName = (guchar *) sender_name_in,
		msg_listing->ucSenderNameLen = strlen(sender_name_in);

		msg_listing->pucSenderAddress = (guchar *) sender_address_in,
		msg_listing->ucSenderAddressLen = strlen(sender_address_in);

		msg_listing->pucReplytoAddress = (guchar *) reply_to_address_in,
		msg_listing->ucReplytoAddressLen = strlen(reply_to_address_in);

		msg_listing->pucRecipientName = (guchar *) recipient_name_in,
		msg_listing->ucRecipientNameLen = strlen(recipient_name_in);

		msg_listing->pucRecipientAddress = (guchar *) recipient_address_in,
		msg_listing->ucRecipientAddressLen = strlen(recipient_address_in);

		msg_listing->ucType = type_in;
		msg_listing->ulSize = size_in;
		msg_listing->ucText = text_in;
		msg_listing->ucReceptionStatus = reception_status_in;
		msg_listing->ulAttachSize = attach_size_in;
		msg_listing->ucPriority = priority_in;
		msg_listing->ucRead = read_in;
		msg_listing->ucSent = sent_in;
		msg_listing->ucProtected = protect_in;

		/* call the callback handler function */
		result = BT_APPL_MAP_List_Msg_IND_Callback(
						*device_address,
						strlen(service_name_in),
						(guchar *) service_name_in,
						msg_listing);
		g_free(msg_listing);
	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_MAP_List_Msg_IND",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestMapListMsgDataComplete"
 */
gboolean test_object_test_map_list_msg_data_complete(TestObject *object,
							GByteArray *addr_in,
							gchar *service_name_in,
							guint32 size_in,
							guint8 status_in,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

	DEBUG("address length %d", addr_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *) &addr_in->data[0];

		DEBUG("service_name %s", service_name_in);

		/* call the callback handler function */
		result = BT_APPL_MAP_List_Msg_Comp_IND_Callback(
						*device_address,
						strlen(service_name_in),
						(guchar *) service_name_in,
						size_in,
						status_in);
	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_List_Msg_Comp_IND",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call for "TestBtApplAvpGetMetadataAttrCfm"
 */
gboolean test_object_test_bt_appl_avp_get_metadata_attr_cfm(TestObject *object,
						GByteArray *addr_in,
						gboolean use_dummy_api_in,
						guint8 dbus_error_in,
						guint8 status_in,
						guint8 num_attributes_in,
						GByteArray *data_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);
	g_assert(data_in != NULL);

#ifdef USE_DUMMY_API

	DEBUG("address length %d", addr_in->len);
	DEBUG("data length %d", data_in->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	if (result == BT_APPL_SUCCESS) {

		/* set individual DUMMY_API settings */

		DEBUG("use DUMMY_API function: %s",
			use_dummy_api_in == TRUE ? "yes" : "no");

		DEBUG("set D-bus error to: %s",
			lookup_bthresult_str((BTHRESULT) dbus_error_in));

		/* change the DUMMY_API setting */
		dummy_change_dummy_api_settings(FUNC_AVP_Get_Metadata_Attr_REQ,
							use_dummy_api_in,
							dbus_error_in);
	}

	if ((result == BT_APPL_SUCCESS) && (use_dummy_api_in == TRUE)) {
		/* only save the test data when the DUMMY API is being used */

		struct avp_get_metadata_attr_cfm_parms_s *cfm_test_data;

		/* create a record */
		cfm_test_data = (struct avp_get_metadata_attr_cfm_parms_s *)
				g_malloc0(sizeof(struct avp_get_metadata_attr_cfm_parms_s));

		g_assert(cfm_test_data != NULL);

		cfm_test_data->status = status_in;
		cfm_test_data->num_attributes = num_attributes_in;

		/* create a new byte array for this data */
		cfm_test_data->data = g_byte_array_new();

		/* copy the data into the GByteArray for storage */
		cfm_test_data->data = g_byte_array_append(cfm_test_data->data,
							&data_in->data[0],
							data_in->len);

		utils_test_queue_add_list_entry(&object->tql_cfm,
						ID_AVP_Get_Metadata_Attr_CFM,
						&addr_in->data[0],
						cfm_test_data);
	}

#else
	/*
         * spoof an ALPS error code as the DUMMY_API is not compiled in
	 */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
					"Test_BT_APPL_AVP_Get_Metadata_Attr_CFM",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestAvpGetMediaPlayerItemCfm"
 */
gboolean test_object_test_avp_get_media_player_item_cfm(TestObject *object,
						GByteArray *addr_in,
						gboolean use_dummy_api_in,
						guint8 dbus_error_in,
						guint8 status_in,
						guint32 num_items_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

#ifdef USE_DUMMY_API

	DEBUG("status %u", status_in);
	DEBUG("num_items %d", num_items_in);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	if (result == BT_APPL_SUCCESS) {

		/* set individual DUMMY_API settings */

		DEBUG("use DUMMY_API function: %s",
			use_dummy_api_in == TRUE ? "yes" : "no");

		DEBUG("set D-bus error to: %s",
			lookup_bthresult_str((BTHRESULT) dbus_error_in));

		/* change the DUMMY_API setting */
		dummy_change_dummy_api_settings(
					FUNC_AVP_Get_Media_Player_Item_REQ,
					use_dummy_api_in,
					dbus_error_in);
	}

	if ((result == BT_APPL_SUCCESS) && (use_dummy_api_in == TRUE)) {
		/* only save the test data when the DUMMY API is being used */

		struct avp_get_media_player_item_cfm_parms_s *cfm_test_data;

		/* create a record */
		cfm_test_data = (struct avp_get_media_player_item_cfm_parms_s *)
				g_malloc0(sizeof(struct avp_get_media_player_item_cfm_parms_s));

		g_assert(cfm_test_data != NULL);

		cfm_test_data->status = status_in;
		cfm_test_data->num_items = num_items_in;

		utils_test_queue_add_list_entry(&object->tql_cfm,
					ID_AVP_Get_Media_Player_Item_CFM,
					&addr_in->data[0],
					cfm_test_data);
	}

#else
	/*
         * spoof an ALPS error code as the DUMMY_API is not compiled in
	 */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
				"Test_BT_APPL_AVP_Get_Media_Player_Item_CFM",
				error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "TestAvpGetMediaPlayerItemInd"
 */
gboolean test_object_test_avp_get_media_player_item_ind(TestObject *object,
						GByteArray *addr_in,
						guint8 status_in,
						GByteArray *data_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);
	g_assert(data_in != NULL);

#ifdef USE_DUMMY_API

	DEBUG("status %u", status_in);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		struct avp_get_media_player_item_ind_parms_s *ind_test_data;

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);


		/* create a record */
		ind_test_data = (struct avp_get_media_player_item_ind_parms_s *)
				g_malloc0(sizeof(struct avp_get_media_player_item_ind_parms_s));

		g_assert(ind_test_data != NULL);

		ind_test_data->status = status_in;

		/* create a new byte array for this data */
		ind_test_data->data = g_byte_array_new();
		g_assert(ind_test_data->data != NULL);

		DEBUG_DUMP_ARRAY(data_in->len, &data_in->data[0]);

		/* copy the data into the GByteArray for storage */
		ind_test_data->data = g_byte_array_append(ind_test_data->data,
							&data_in->data[0],
							data_in->len);

		utils_test_queue_add_list_entry(&object->tql_cfm,
					ID_AVP_Get_Media_Player_Item_IND,
					&addr_in->data[0],
					ind_test_data);
	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

#else
	/*
         * spoof an ALPS error code as the DUMMY_API is not compiled in
	 */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
				"Test_BT_APPL_AVP_Get_Media_Player_Item_CFM",
				error);

	DEBUG_FUNC("Exited");

	return return_code;
}

gboolean test_object_test_bt_appl_authentication_ind(TestObject *object,
						GByteArray *addr_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr_in != NULL);

#ifdef USE_DUMMY_API

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr_in->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr_in->data[0], addr_in->data[1],
			 addr_in->data[2], addr_in->data[3],
			 addr_in->data[4], addr_in->data[5]);

		BT_APPL_Authentication_IND_Callback_DUMMY(&addr_in->data[0]);
	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

#else
	/*
         * spoof an ALPS error code as the DUMMY_API is not compiled in
	 */
	result = BT_APPL_ERROR_NOT_SUPPORTED;
#endif

	return_code = report_any_error_to_dbus(result,
				"Test_BT_APPL_BT_Authenication_IND",
				error);

	DEBUG_FUNC("Exited");

	return return_code;
}

#endif
